<?php
session_start();
include 'includes/db.php';

$error = "";
$success = "";
$show_form = true;

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $email = trim($_POST['email']);
    $new_password = trim($_POST['new_password'] ?? '');
    $confirm_password = trim($_POST['confirm_password'] ?? '');
    
    // Step 1: User submits email to check existence
    if (isset($_POST['email']) && !isset($_POST['new_password'])) {
        if (empty($email)) {
            $error = "Please enter your email.";
        } else {
            $sql = "SELECT id, name FROM users WHERE email = ?";
            $stmt = $conn->prepare($sql);
            $stmt->bind_param("s", $email);
            $stmt->execute();
            $result = $stmt->get_result();
            if ($result->num_rows === 1) {
                $user = $result->fetch_assoc();
                $show_form = false; // Show password reset form
                $_SESSION['reset_user_id'] = $user['id'];
                $_SESSION['reset_user_name'] = $user['name'];
            } else {
                $error = "No account found with this email.";
            }
        }
    }
    
    // Step 2: User submits new password
    if (isset($_POST['new_password'])) {
        if (empty($new_password) || empty($confirm_password)) {
            $error = "Please fill in all password fields.";
        } elseif ($new_password !== $confirm_password) {
            $error = "Passwords do not match.";
        } elseif (!isset($_SESSION['reset_user_id'])) {
            $error = "Session expired. Please try again.";
        } else {
            $hashed_password = password_hash($new_password, PASSWORD_DEFAULT);
            $user_id = $_SESSION['reset_user_id'];
            $sql = "UPDATE users SET password = ? WHERE id = ?";
            $stmt = $conn->prepare($sql);
            $stmt->bind_param("si", $hashed_password, $user_id);
            if ($stmt->execute()) {
                $success = "Password reset successfully! You can now <a href='login.php'>login</a>.";
                unset($_SESSION['reset_user_id']);
                unset($_SESSION['reset_user_name']);
                $show_form = false;
            } else {
                $error = "Failed to reset password. Try again.";
            }
        }
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
<meta charset="UTF-8">
<meta name="viewport" content="width=device-width, initial-scale=1.0">
<title>MYVTU - Forgot Password</title>
<link rel="stylesheet" href="css/style.css">
<style>
    body {
        font-family: Arial, sans-serif;
        background-color: #f4f6f8;
        display: flex;
        flex-direction: column;
        min-height: 100vh;
        align-items: center;
        justify-content: center;
    }
    .forgot-container {
        background: #fff;
        padding: 40px 30px;
        border-radius: 10px;
        box-shadow: 0 4px 8px rgba(0,0,0,0.1);
        text-align: center;
        width: 300px;
    }
    .forgot-container img {
        width: 100px;
        margin-bottom: 20px;
    }
    .forgot-container h2 {
        color: #203a43;
        margin-bottom: 20px;
    }
    input[type="email"], input[type="password"] {
        width: 100%;
        padding: 12px;
        margin: 10px 0;
        border: 1px solid #ccc;
        border-radius: 6px;
    }
    input[type="submit"] {
        width: 100%;
        padding: 12px;
        background-color: #203a43;
        color: #f4f6f8;
        border: none;
        border-radius: 6px;
        font-weight: bold;
        cursor: pointer;
    }
    input[type="submit"]:hover {
        background-color: #ff6600;
    }
    .error { color: red; margin-bottom: 15px; }
    .success { color: green; margin-bottom: 15px; }
    a.back-link {
        display: block;
        margin-top: 15px;
        text-decoration: none;
        color: #203a43;
    }
    a.back-link:hover { color: #ff6600; }
</style>
</head>
<body>

<div class="forgot-container">
    <img src="assets/logo.png" alt="MYVTU Logo">
    <h2>Forgot Password</h2>

    <?php if ($error != ""): ?>
        <div class="error"><?php echo htmlspecialchars($error); ?></div>
    <?php endif; ?>

    <?php if ($success != ""): ?>
        <div class="success"><?php echo $success; ?></div>
    <?php endif; ?>

    <?php if ($show_form): ?>
        <form method="POST" action="">
            <input type="email" name="email" placeholder="Enter your email" required>
            <input type="submit" value="Verify Email">
        </form>
    <?php elseif (isset($_SESSION['reset_user_id'])): ?>
        <p>Hello, <?php echo htmlspecialchars($_SESSION['reset_user_name']); ?>. Enter a new password.</p>
        <form method="POST" action="">
            <input type="password" name="new_password" placeholder="New Password" required>
            <input type="password" name="confirm_password" placeholder="Confirm Password" required>
            <input type="submit" value="Reset Password">
        </form>
    <?php endif; ?>

    <a class="back-link" href="login.php">Back to Login</a>
</div>

</body>
</html>